<?php
require_auth();
require __DIR__ . '/../db.php';
require __DIR__ . '/../helpers.php';

header('Content-Type: application/json; charset=utf-8');

function valid_date($value) {
    return is_string($value) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $value) && strtotime($value) !== false;
}

function period_from_request($prefix = '') {
    $modeKey = $prefix . 'mode';
    $mode = isset($_GET[$modeKey]) ? (string)$_GET[$modeKey] : 'month';
    if (!in_array($mode, ['day', 'week', 'month', 'custom'], true)) {
        $mode = 'month';
    }

    if ($mode === 'day') {
        $day = isset($_GET[$prefix . 'day']) ? (string)$_GET[$prefix . 'day'] : date('Y-m-d');
        if (!valid_date($day)) {
            $day = date('Y-m-d');
        }
        return [
            'mode' => 'day',
            'ini' => $day,
            'fim' => $day,
            'label' => date('d/m/Y', strtotime($day)),
        ];
    }

    if ($mode === 'week') {
        $week = isset($_GET[$prefix . 'week']) ? (string)$_GET[$prefix . 'week'] : date('o-\\WW');
        if (!preg_match('/^(\d{4})-W(\d{2})$/', $week, $m)) {
            $week = date('o-\\WW');
            preg_match('/^(\d{4})-W(\d{2})$/', $week, $m);
        }
        $year = (int)$m[1];
        $num = (int)$m[2];
        $dt = new DateTime();
        $dt->setISODate($year, $num);
        $ini = $dt->format('Y-m-d');
        $dt->modify('+6 days');
        $fim = $dt->format('Y-m-d');
        return [
            'mode' => 'week',
            'ini' => $ini,
            'fim' => $fim,
            'label' => date('d/m', strtotime($ini)) . ' a ' . date('d/m', strtotime($fim)),
        ];
    }

    if ($mode === 'custom') {
        $ini = isset($_GET[$prefix . 'dt_ini']) ? (string)$_GET[$prefix . 'dt_ini'] : date('Y-m-01');
        $fim = isset($_GET[$prefix . 'dt_fim']) ? (string)$_GET[$prefix . 'dt_fim'] : date('Y-m-d');
        if (!valid_date($ini) || !valid_date($fim)) {
            $ini = date('Y-m-01');
            $fim = date('Y-m-d');
        }
        if ($fim < $ini) {
            $tmp = $ini;
            $ini = $fim;
            $fim = $tmp;
        }
        return [
            'mode' => 'custom',
            'ini' => $ini,
            'fim' => $fim,
            'label' => date('d/m/Y', strtotime($ini)) . ' a ' . date('d/m/Y', strtotime($fim)),
        ];
    }

    $mes = isset($_GET[$prefix . 'mes']) ? (string)$_GET[$prefix . 'mes'] : date('Y-m');
    if (!preg_match('/^\d{4}-\d{2}$/', $mes)) {
        $mes = date('Y-m');
    }
    $ini = $mes . '-01';
    $fim = date('Y-m-t', strtotime($ini));

    return [
        'mode' => 'month',
        'ini' => $ini,
        'fim' => $fim,
        'label' => date('m/Y', strtotime($ini)),
    ];
}

function previous_period($ini, $fim) {
    $days = (int)floor((strtotime($fim) - strtotime($ini)) / 86400) + 1;
    $prevFim = date('Y-m-d', strtotime($ini . ' -1 day'));
    $prevIni = date('Y-m-d', strtotime($prevFim . ' -' . ($days - 1) . ' day'));
    return [
        'mode' => 'custom',
        'ini' => $prevIni,
        'fim' => $prevFim,
        'label' => date('d/m/Y', strtotime($prevIni)) . ' a ' . date('d/m/Y', strtotime($prevFim)),
    ];
}

function query_kpis(PDO $pdo, $ini, $fim) {
    $st = $pdo->prepare("SELECT
      COUNT(*) AS registros,
      COALESCE(SUM(pedido),0) AS pedido_total,
      COALESCE(SUM(realizado),0) AS realizado_total,
      COALESCE(SUM(perda_total),0) AS perda_total,
      COALESCE(AVG(perc_perda),0) AS perc_media,
      COALESCE(SUM(valor_total_previsto),0) AS valor_previsto,
      COALESCE(SUM(valor_total_realizado),0) AS valor_realizado
    FROM bi_perdas
    WHERE dt_producao BETWEEN ? AND ?");
    $st->execute([$ini, $fim]);
    $k = $st->fetch() ?: [];

    $pedidoTotal = (int)($k['pedido_total'] ?? 0);
    $realizadoTotal = (int)($k['realizado_total'] ?? 0);
    $saldoUnidades = $realizadoTotal - $pedidoTotal;
    $saldoPerc = $pedidoTotal > 0 ? round(($saldoUnidades / $pedidoTotal) * 100, 2) : 0;
    $valorPrevisto = (float)($k['valor_previsto'] ?? 0);
    $valorRealizado = (float)($k['valor_realizado'] ?? 0);
    $saldoFinanceiro = $valorRealizado - $valorPrevisto;

    return [
        'registros' => (int)($k['registros'] ?? 0),
        'pedido_total' => $pedidoTotal,
        'realizado_total' => $realizadoTotal,
        'saldo_unidades' => $saldoUnidades,
        'saldo_perc' => $saldoPerc,
        'perda_total' => (int)($k['perda_total'] ?? 0),
        'perc_media' => round((float)($k['perc_media'] ?? 0), 2),
        'valor_previsto' => $valorPrevisto,
        'valor_realizado' => $valorRealizado,
        'saldo_financeiro' => $saldoFinanceiro,
        'valor_previsto_fmt' => money($valorPrevisto),
        'valor_realizado_fmt' => money($valorRealizado),
        'saldo_financeiro_fmt' => money($saldoFinanceiro),
    ];
}

function diff_metrics($a, $b) {
    $keys = ['registros', 'pedido_total', 'realizado_total', 'saldo_unidades', 'perda_total', 'perc_media', 'valor_previsto', 'valor_realizado', 'saldo_financeiro'];
    $out = [];
    foreach ($keys as $k) {
        $va = (float)($a[$k] ?? 0);
        $vb = (float)($b[$k] ?? 0);
        $delta = $va - $vb;
        $pct = $vb != 0 ? round(($delta / $vb) * 100, 2) : null;
        $out[$k] = ['delta' => $delta, 'pct' => $pct];
    }
    return $out;
}

$period = period_from_request('');
$compareEnabled = !empty($_GET['compare']) && (string)$_GET['compare'] === '1';
$comparePeriod = null;
if ($compareEnabled) {
    $hasCompareInputs = isset($_GET['compare_mode']) || isset($_GET['compare_mes']) || isset($_GET['compare_day']) || isset($_GET['compare_week']) || isset($_GET['compare_dt_ini']) || isset($_GET['compare_dt_fim']);
    if ($hasCompareInputs) {
        $comparePeriod = period_from_request('compare_');
    } else {
        $comparePeriod = previous_period($period['ini'], $period['fim']);
    }
}

$pdo = db();
$kpis = query_kpis($pdo, $period['ini'], $period['fim']);
$kpisCompare = null;
$kpisDiff = null;
if ($comparePeriod) {
    $kpisCompare = query_kpis($pdo, $comparePeriod['ini'], $comparePeriod['fim']);
    $kpisDiff = diff_metrics($kpis, $kpisCompare);
}

$motivosCols = [
  'perda_setup' => 'Setup',
  'abaulada' => 'Abaulada',
  'impressao_nao_conforme' => 'Impressao nao conforme',
  'materia_prima' => 'Materia prima',
  'acabou_cola' => 'Acabou cola',
  'acabou_tinta' => 'Acabou tinta',
  'delaminacao' => 'Delaminacao',
  'fora_de_esquadro' => 'Fora de esquadro',
  'rasgada' => 'Rasgada',
  'enroscou' => 'Enroscou',
  'outros' => 'Outros',
];

$parts = [];
foreach ($motivosCols as $c => $label) {
    $parts[] = "COALESCE(SUM($c),0) AS `$c`";
}
$st = $pdo->prepare('SELECT ' . implode(',', $parts) . ' FROM bi_perdas WHERE dt_producao BETWEEN ? AND ?');
$st->execute([$period['ini'], $period['fim']]);
$m = $st->fetch() ?: [];

$combined = [];
foreach ($motivosCols as $c => $label) {
    $combined[] = [$label, (int)($m[$c] ?? 0)];
}
usort($combined, function ($a, $b) {
    return $b[1] <=> $a[1];
});
$labels = [];
$values = [];
foreach ($combined as $pair) {
    $labels[] = $pair[0];
    $values[] = $pair[1];
}

$st = $pdo->prepare("SELECT cliente, COALESCE(SUM(perda_total),0) AS perda
  FROM bi_perdas
  WHERE dt_producao BETWEEN ? AND ?
  GROUP BY cliente
  ORDER BY perda DESC
  LIMIT 8");
$st->execute([$period['ini'], $period['fim']]);
$rows = $st->fetchAll();
$cl = ['labels' => [], 'values' => []];
foreach ($rows as $r) {
    $cl['labels'][] = $r['cliente'] ?: '(Sem cliente)';
    $cl['values'][] = (int)$r['perda'];
}

$st = $pdo->prepare("SELECT
    COALESCE(material, '') AS material,
    COALESCE(SUM(perda_total),0) AS perda,
    COALESCE(SUM(pedido),0) AS pedido,
    COALESCE(SUM(realizado),0) AS realizado
  FROM bi_perdas
  WHERE dt_producao BETWEEN ? AND ?
  GROUP BY material
  ORDER BY perda DESC
  LIMIT 8");
$st->execute([$period['ini'], $period['fim']]);
$materialRows = $st->fetchAll();
$material = ['labels' => [], 'values' => [], 'rows' => []];
foreach ($materialRows as $r) {
    $label = trim((string)$r['material']) !== '' ? (string)$r['material'] : '(Sem material)';
    $pedido = (int)$r['pedido'];
    $realizado = (int)$r['realizado'];
    $dif = $realizado - $pedido;
    $material['labels'][] = $label;
    $material['values'][] = (int)$r['perda'];
    $material['rows'][] = [
        'material' => $label,
        'perda' => (int)$r['perda'],
        'pedido' => $pedido,
        'realizado' => $realizado,
        'diferenca' => $dif,
    ];
}

$st = $pdo->prepare("SELECT
    cliente,
    COALESCE(SUM(pedido),0) AS pedido_total,
    COALESCE(SUM(realizado),0) AS realizado_total,
    COALESCE(SUM(realizado - pedido),0) AS diferenca,
    COALESCE(SUM(ABS(realizado - pedido)),0) AS diferenca_abs
  FROM bi_perdas
  WHERE dt_producao BETWEEN ? AND ?
  GROUP BY cliente
  ORDER BY diferenca_abs DESC
  LIMIT 10");
$st->execute([$period['ini'], $period['fim']]);
$clienteDiffRows = $st->fetchAll();

$st = $pdo->prepare("SELECT
    op_trimb,
    cliente,
    COALESCE(SUM(pedido),0) AS pedido_total,
    COALESCE(SUM(realizado),0) AS realizado_total,
    COALESCE(SUM(realizado - pedido),0) AS diferenca
  FROM bi_perdas
  WHERE dt_producao BETWEEN ? AND ?
  GROUP BY op_trimb, cliente
  ORDER BY ABS(COALESCE(SUM(realizado - pedido),0)) DESC
  LIMIT 10");
$st->execute([$period['ini'], $period['fim']]);
$pedidoDiffRows = $st->fetchAll();

$st = $pdo->prepare("SELECT
    dt_producao,
    COALESCE(SUM(perda_total),0) AS perda,
    COALESCE(SUM(realizado - pedido),0) AS saldo
  FROM bi_perdas
  WHERE dt_producao BETWEEN ? AND ?
  GROUP BY dt_producao
  ORDER BY dt_producao ASC");
$st->execute([$period['ini'], $period['fim']]);
$serieRows = $st->fetchAll();
$serie = ['labels' => [], 'perda' => [], 'saldo' => []];
foreach ($serieRows as $r) {
    $serie['labels'][] = date('d/m', strtotime($r['dt_producao']));
    $serie['perda'][] = (int)$r['perda'];
    $serie['saldo'][] = (int)$r['saldo'];
}

echo json_encode([
  'periodo' => $period,
  'compare_enabled' => $compareEnabled,
  'compare_periodo' => $comparePeriod,
  'kpis' => $kpis,
  'kpis_compare' => $kpisCompare,
  'kpis_diff' => $kpisDiff,
  'motivos' => ['labels' => $labels, 'values' => $values],
  'clientes' => $cl,
  'material' => $material,
  'cliente_diferenca' => $clienteDiffRows,
  'pedido_diferenca' => $pedidoDiffRows,
  'serie_diaria' => $serie,
], JSON_UNESCAPED_UNICODE);
